var __defProp = Object.defineProperty;
var __getOwnPropNames = Object.getOwnPropertyNames;
var __require = /* @__PURE__ */ ((x) => typeof require !== "undefined" ? require : typeof Proxy !== "undefined" ? new Proxy(x, {
  get: (a, b) => (typeof require !== "undefined" ? require : a)[b]
}) : x)(function(x) {
  if (typeof require !== "undefined") return require.apply(this, arguments);
  throw Error('Dynamic require of "' + x + '" is not supported');
});
var __esm = (fn, res) => function __init() {
  return fn && (res = (0, fn[__getOwnPropNames(fn)[0]])(fn = 0)), res;
};
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};

// vite.config.ts
var vite_config_exports = {};
__export(vite_config_exports, {
  default: () => vite_config_default
});
import { defineConfig } from "vite";
import react from "@vitejs/plugin-react";
import path from "path";
import runtimeErrorOverlay from "@replit/vite-plugin-runtime-error-modal";
var vite_config_default;
var init_vite_config = __esm({
  async "vite.config.ts"() {
    "use strict";
    vite_config_default = defineConfig({
      plugins: [
        react(),
        runtimeErrorOverlay(),
        ...process.env.NODE_ENV !== "production" && process.env.REPL_ID !== void 0 ? [
          await import("@replit/vite-plugin-cartographer").then(
            (m) => m.cartographer()
          )
        ] : []
      ],
      resolve: {
        alias: {
          "@": path.resolve(import.meta.dirname, "client", "src"),
          "@shared": path.resolve(import.meta.dirname, "shared"),
          "@assets": path.resolve(import.meta.dirname, "attached_assets")
        }
      },
      root: path.resolve(import.meta.dirname, "client"),
      build: {
        outDir: path.resolve(import.meta.dirname, "dist/public"),
        emptyOutDir: true
      }
    });
  }
});

// server/vite.ts
var vite_exports = {};
__export(vite_exports, {
  log: () => log,
  serveStatic: () => serveStatic,
  setupVite: () => setupVite
});
import express from "express";
import fs from "fs";
import path2 from "path";
import { nanoid } from "nanoid";
function log(message, source = "express") {
  const formattedTime = (/* @__PURE__ */ new Date()).toLocaleTimeString("en-US", {
    hour: "numeric",
    minute: "2-digit",
    second: "2-digit",
    hour12: true
  });
  console.log(`${formattedTime} [${source}] ${message}`);
}
function serveStatic(app2) {
  const distPath = path2.resolve(__dirname, "../../dist/public");
  if (!fs.existsSync(distPath)) {
    throw new Error(
      `Could not find the build directory: ${distPath}, make sure to build the client first`
    );
  }
  app2.use(express.static(distPath));
  app2.use("*", (_req, res) => {
    res.sendFile(path2.resolve(distPath, "index.html"));
  });
}
async function setupVite(app2, server) {
  const { createServer: createViteServer, createLogger } = await import("vite");
  const viteConfig = (await init_vite_config().then(() => vite_config_exports)).default;
  const viteLogger = createLogger();
  const serverOptions = {
    middlewareMode: true,
    hmr: { server },
    allowedHosts: true
  };
  const vite = await createViteServer({
    ...viteConfig,
    configFile: false,
    customLogger: {
      ...viteLogger,
      error: (msg, options) => {
        viteLogger.error(msg, options);
        process.exit(1);
      }
    },
    server: serverOptions,
    appType: "custom"
  });
  app2.use(vite.middlewares);
  app2.use("*", async (req, res, next) => {
    const url = req.originalUrl;
    try {
      const clientTemplate = path2.resolve(
        __dirname,
        "..",
        "client",
        "index.html"
      );
      let template = await fs.promises.readFile(clientTemplate, "utf-8");
      template = template.replace(
        `src="/src/main.tsx"`,
        `src="/src/main.tsx?v=${nanoid()}"`
      );
      const page = await vite.transformIndexHtml(url, template);
      res.status(200).set({ "Content-Type": "text/html" }).end(page);
    } catch (e) {
      vite.ssrFixStacktrace(e);
      next(e);
    }
  });
}
var init_vite = __esm({
  "server/vite.ts"() {
    "use strict";
  }
});

// server/index.ts
import express2 from "express";

// server/routes.ts
import session from "express-session";
import { createServer } from "http";

// shared/schema.ts
var schema_exports = {};
__export(schema_exports, {
  cartItems: () => cartItems,
  cartItemsRelations: () => cartItemsRelations,
  categories: () => categories,
  categoriesRelations: () => categoriesRelations,
  insertCartItemSchema: () => insertCartItemSchema,
  insertCategorySchema: () => insertCategorySchema,
  insertOrderItemSchema: () => insertOrderItemSchema,
  insertOrderSchema: () => insertOrderSchema,
  insertProductSchema: () => insertProductSchema,
  insertUserSchema: () => insertUserSchema,
  loginSchema: () => loginSchema,
  orderItems: () => orderItems,
  orderItemsRelations: () => orderItemsRelations,
  orders: () => orders,
  ordersRelations: () => ordersRelations,
  products: () => products,
  productsRelations: () => productsRelations,
  registerSchema: () => registerSchema,
  users: () => users,
  usersRelations: () => usersRelations
});
import { pgTable, text, serial, integer, boolean, decimal, timestamp } from "drizzle-orm/pg-core";
import { relations } from "drizzle-orm";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";
var users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: text("username").notNull().unique(),
  password: text("password").notNull(),
  email: text("email").notNull().unique(),
  isAdmin: boolean("is_admin").default(false),
  isVerified: boolean("is_verified").default(false),
  verificationCode: text("verification_code"),
  verificationCodeExpiry: timestamp("verification_code_expiry"),
  resetCode: text("reset_code"),
  resetCodeExpiry: timestamp("reset_code_expiry"),
  createdAt: timestamp("created_at").defaultNow()
});
var categories = pgTable("categories", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  slug: text("slug").notNull().unique(),
  description: text("description")
});
var products = pgTable("products", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  description: text("description").notNull(),
  categoryId: integer("category_id").references(() => categories.id),
  priceUSD: decimal("price_usd", { precision: 10, scale: 2 }).notNull(),
  originalPriceUSD: decimal("original_price_usd", { precision: 10, scale: 2 }),
  imageUrl: text("image_url").notNull(),
  digitalContent: text("digital_content"),
  // License keys, download links, etc.
  isActive: boolean("is_active").default(true),
  stock: integer("stock").default(-1),
  // -1 for unlimited
  deliveryInstructions: text("delivery_instructions"),
  createdAt: timestamp("created_at").defaultNow()
});
var orders = pgTable("orders", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").references(() => users.id),
  customerEmail: text("customer_email").notNull(),
  customerPhone: text("customer_phone"),
  paypalOrderId: text("paypal_order_id"),
  status: text("status").notNull().default("pending"),
  // pending, completed, failed, delivered
  totalUSD: decimal("total_usd", { precision: 10, scale: 2 }).notNull(),
  currency: text("currency").notNull().default("USD"),
  createdAt: timestamp("created_at").defaultNow(),
  deliveredAt: timestamp("delivered_at")
});
var orderItems = pgTable("order_items", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").references(() => orders.id),
  productId: integer("product_id").references(() => products.id),
  quantity: integer("quantity").notNull(),
  priceUSD: decimal("price_usd", { precision: 10, scale: 2 }).notNull(),
  digitalContent: text("digital_content"),
  // Delivered content
  pubgPlayerId: text("pubg_player_id")
  // PUBG Player ID for UC delivery
});
var cartItems = pgTable("cart_items", {
  id: serial("id").primaryKey(),
  sessionId: text("session_id").notNull(),
  productId: integer("product_id").references(() => products.id),
  quantity: integer("quantity").notNull(),
  createdAt: timestamp("created_at").defaultNow()
});
var insertUserSchema = createInsertSchema(users).pick({
  username: true,
  password: true,
  email: true,
  isAdmin: true,
  isVerified: true,
  verificationCode: true,
  verificationCodeExpiry: true,
  resetCode: true,
  resetCodeExpiry: true
});
var registerSchema = z.object({
  username: z.string().min(3, "Username must be at least 3 characters"),
  email: z.string().email("Invalid email address"),
  password: z.string().min(8, "Password must be at least 8 characters"),
  confirmPassword: z.string().min(1, "Please confirm your password")
}).refine((data) => data.password === data.confirmPassword, {
  message: "Passwords don't match",
  path: ["confirmPassword"]
});
var loginSchema = z.object({
  email: z.string().email("Please enter a valid email address"),
  password: z.string().min(1, "Password is required")
});
var insertCategorySchema = createInsertSchema(categories).pick({
  name: true,
  slug: true,
  description: true
});
var insertProductSchema = createInsertSchema(products).pick({
  name: true,
  description: true,
  categoryId: true,
  priceUSD: true,
  originalPriceUSD: true,
  imageUrl: true,
  digitalContent: true,
  stock: true,
  deliveryInstructions: true
});
var insertOrderSchema = createInsertSchema(orders).pick({
  customerEmail: true,
  customerPhone: true,
  totalUSD: true,
  currency: true,
  status: true
});
var insertOrderItemSchema = createInsertSchema(orderItems).pick({
  orderId: true,
  productId: true,
  quantity: true,
  priceUSD: true,
  pubgPlayerId: true
});
var insertCartItemSchema = createInsertSchema(cartItems).pick({
  sessionId: true,
  productId: true,
  quantity: true
});
var usersRelations = relations(users, ({ many }) => ({
  orders: many(orders)
}));
var categoriesRelations = relations(categories, ({ many }) => ({
  products: many(products)
}));
var productsRelations = relations(products, ({ one, many }) => ({
  category: one(categories, {
    fields: [products.categoryId],
    references: [categories.id]
  }),
  orderItems: many(orderItems),
  cartItems: many(cartItems)
}));
var ordersRelations = relations(orders, ({ one, many }) => ({
  user: one(users, {
    fields: [orders.userId],
    references: [users.id]
  }),
  items: many(orderItems)
}));
var orderItemsRelations = relations(orderItems, ({ one }) => ({
  order: one(orders, {
    fields: [orderItems.orderId],
    references: [orders.id]
  }),
  product: one(products, {
    fields: [orderItems.productId],
    references: [products.id]
  })
}));
var cartItemsRelations = relations(cartItems, ({ one }) => ({
  product: one(products, {
    fields: [cartItems.productId],
    references: [products.id]
  })
}));

// server/db.ts
import dotenv from "dotenv";
import mysql from "mysql2/promise";
import { drizzle } from "drizzle-orm/mysql2";
dotenv.config();
if (!process.env.DATABASE_URL) {
  throw new Error(
    "DATABASE_URL must be set. Did you forget to provision a database?"
  );
}
var pool = mysql.createPool(process.env.DATABASE_URL);
var db = drizzle(pool, { schema: schema_exports, mode: "default" });

// server/storage.ts
import { eq, and } from "drizzle-orm";
var DatabaseStorage = class {
  async getUser(id) {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user || void 0;
  }
  async getUserByUsername(username) {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user || void 0;
  }
  async getUserByEmail(email) {
    const [user] = await db.select().from(users).where(eq(users.email, email));
    return user || void 0;
  }
  async createUser(insertUser) {
    const [user] = await db.insert(users).values(insertUser).returning();
    return user;
  }
  async getUserByVerificationCode(code) {
    const [user] = await db.select().from(users).where(eq(users.verificationCode, code));
    return user || void 0;
  }
  async getUserByResetCode(code) {
    const [user] = await db.select().from(users).where(eq(users.resetCode, code));
    return user || void 0;
  }
  async updateUser(id, updates) {
    const [user] = await db.update(users).set(updates).where(eq(users.id, id)).returning();
    return user || void 0;
  }
  async getCategories() {
    return await db.select().from(categories);
  }
  async getCategory(id) {
    const [category] = await db.select().from(categories).where(eq(categories.id, id));
    return category || void 0;
  }
  async getCategoryBySlug(slug) {
    const [category] = await db.select().from(categories).where(eq(categories.slug, slug));
    return category || void 0;
  }
  async createCategory(insertCategory) {
    const [category] = await db.insert(categories).values(insertCategory).returning();
    return category;
  }
  async updateCategory(id, updates) {
    const [category] = await db.update(categories).set(updates).where(eq(categories.id, id)).returning();
    return category || void 0;
  }
  async deleteCategory(id) {
    const result = await db.delete(categories).where(eq(categories.id, id));
    return (result.rowCount || 0) > 0;
  }
  async getProducts() {
    return await db.query.products.findMany({
      with: {
        category: true
      }
    });
  }
  async getProductsByCategory(categoryId) {
    return await db.query.products.findMany({
      where: eq(products.categoryId, categoryId),
      with: {
        category: true
      }
    });
  }
  async getProduct(id) {
    const product = await db.query.products.findFirst({
      where: eq(products.id, id),
      with: {
        category: true
      }
    });
    return product || void 0;
  }
  async getFeaturedProducts() {
    return await db.query.products.findMany({
      where: eq(products.isActive, true),
      with: {
        category: true
      },
      limit: 6
    });
  }
  async createProduct(insertProduct) {
    const [product] = await db.insert(products).values(insertProduct).returning();
    return product;
  }
  async updateProduct(id, updates) {
    const [product] = await db.update(products).set(updates).where(eq(products.id, id)).returning();
    return product || void 0;
  }
  async deleteProduct(id) {
    const result = await db.delete(products).where(eq(products.id, id));
    return (result.rowCount || 0) > 0;
  }
  async getOrders() {
    const orders2 = await db.query.orders.findMany({
      with: {
        items: {
          with: {
            product: true
          }
        }
      }
    });
    return orders2.map((order) => ({
      ...order,
      items: order.items.filter((item) => item.product !== null).map((item) => ({ ...item, product: item.product }))
    }));
  }
  async getOrder(id) {
    const order = await db.query.orders.findFirst({
      where: eq(orders.id, id),
      with: {
        items: {
          with: {
            product: true
          }
        }
      }
    });
    if (!order) return void 0;
    const filteredItems = order.items.filter((item) => item.product !== null).map((item) => ({ ...item, product: item.product }));
    return { ...order, items: filteredItems };
  }
  async getOrderByPaypalId(paypalOrderId) {
    const order = await db.query.orders.findFirst({
      where: eq(orders.paypalOrderId, paypalOrderId),
      with: {
        items: {
          with: {
            product: true
          }
        }
      }
    });
    if (!order) return void 0;
    const filteredItems = order.items.filter((item) => item.product !== null).map((item) => ({ ...item, product: item.product }));
    return { ...order, items: filteredItems };
  }
  async getUserOrders(userId) {
    const userOrders = await db.query.orders.findMany({
      where: eq(orders.userId, userId),
      with: {
        items: {
          with: {
            product: true
          }
        }
      },
      orderBy: (orders2, { desc: desc2 }) => [desc2(orders2.createdAt)]
    });
    return userOrders.map((order) => ({
      ...order,
      items: order.items.filter((item) => item.product !== null).map((item) => ({ ...item, product: item.product }))
    }));
  }
  async createOrder(insertOrder) {
    const [order] = await db.insert(orders).values(insertOrder).returning();
    return order;
  }
  async updateOrderStatus(id, status, paypalOrderId) {
    const updates = { status };
    if (paypalOrderId) {
      updates.paypalOrderId = paypalOrderId;
    }
    if (status === "delivered") {
      updates.deliveredAt = /* @__PURE__ */ new Date();
    }
    const [order] = await db.update(orders).set(updates).where(eq(orders.id, id)).returning();
    return order || void 0;
  }
  async createOrderItem(insertOrderItem) {
    const [orderItem] = await db.insert(orderItems).values(insertOrderItem).returning();
    return orderItem;
  }
  async getOrderItems(orderId) {
    return (await db.query.orderItems.findMany({
      where: eq(orderItems.orderId, orderId),
      with: {
        product: true
      }
    })).filter((item) => item.product !== null).map((item) => ({ ...item, product: item.product }));
  }
  async getCartItems(sessionId) {
    return (await db.query.cartItems.findMany({
      where: eq(cartItems.sessionId, sessionId),
      with: {
        product: true
      }
    })).filter((item) => item.product !== null).map((item) => ({ ...item, product: item.product }));
  }
  async addToCart(insertCartItem) {
    const [existingItem] = await db.select().from(cartItems).where(
      and(
        eq(cartItems.sessionId, insertCartItem.sessionId),
        eq(cartItems.productId, insertCartItem.productId)
      )
    );
    if (existingItem) {
      const [updatedItem] = await db.update(cartItems).set({ quantity: existingItem.quantity + insertCartItem.quantity }).where(eq(cartItems.id, existingItem.id)).returning();
      return updatedItem;
    } else {
      const [newItem] = await db.insert(cartItems).values(insertCartItem).returning();
      return newItem;
    }
  }
  async updateCartItemQuantity(sessionId, productId, quantity) {
    if (quantity <= 0) {
      await this.removeFromCart(sessionId, productId);
      return void 0;
    }
    const [cartItem] = await db.update(cartItems).set({ quantity }).where(
      and(
        eq(cartItems.sessionId, sessionId),
        eq(cartItems.productId, productId)
      )
    ).returning();
    return cartItem || void 0;
  }
  async removeFromCart(sessionId, productId) {
    const result = await db.delete(cartItems).where(
      and(
        eq(cartItems.sessionId, sessionId),
        eq(cartItems.productId, productId)
      )
    );
    return (result.rowCount || 0) > 0;
  }
  async clearCart(sessionId) {
    const result = await db.delete(cartItems).where(eq(cartItems.sessionId, sessionId));
    return (result.rowCount || 0) > 0;
  }
  async mergeCart(guestCartId, userCartId) {
    try {
      const guestItems = await db.select().from(cartItems).where(eq(cartItems.sessionId, guestCartId));
      for (const item of guestItems) {
        const existingUserItems = await db.select().from(cartItems).where(
          and(
            eq(cartItems.sessionId, userCartId),
            eq(cartItems.productId, item.productId)
          )
        );
        if (existingUserItems.length > 0) {
          await db.update(cartItems).set({ quantity: existingUserItems[0].quantity + item.quantity }).where(eq(cartItems.id, existingUserItems[0].id));
        } else {
          await db.insert(cartItems).values({
            sessionId: userCartId,
            productId: item.productId,
            quantity: item.quantity
          });
        }
      }
      await db.delete(cartItems).where(eq(cartItems.sessionId, guestCartId));
      return true;
    } catch (error) {
      console.error("Error merging carts:", error);
      return false;
    }
  }
};
var storage = new DatabaseStorage();

// server/paypal.ts
import {
  Client,
  Environment,
  LogLevel,
  OAuthAuthorizationController,
  OrdersController
} from "@paypal/paypal-server-sdk";
var { PAYPAL_CLIENT_ID, PAYPAL_CLIENT_SECRET } = process.env;
if (!PAYPAL_CLIENT_ID) {
  throw new Error("Missing PAYPAL_CLIENT_ID");
}
if (!PAYPAL_CLIENT_SECRET) {
  throw new Error("Missing PAYPAL_CLIENT_SECRET");
}
var client = new Client({
  clientCredentialsAuthCredentials: {
    oAuthClientId: PAYPAL_CLIENT_ID,
    oAuthClientSecret: PAYPAL_CLIENT_SECRET
  },
  timeout: 0,
  environment: Environment.Production,
  logging: {
    logLevel: LogLevel.Info,
    logRequest: {
      logBody: true
    },
    logResponse: {
      logHeaders: true
    }
  }
});
var ordersController = new OrdersController(client);
var oAuthAuthorizationController = new OAuthAuthorizationController(client);
async function getClientToken() {
  const auth = Buffer.from(
    `${PAYPAL_CLIENT_ID}:${PAYPAL_CLIENT_SECRET}`
  ).toString("base64");
  const { result } = await oAuthAuthorizationController.requestToken(
    {
      authorization: `Basic ${auth}`
    },
    { intent: "sdk_init", response_type: "client_token" }
  );
  return result.accessToken;
}
async function createPaypalOrder(req, res) {
  try {
    const { amount, currency, intent } = req.body;
    if (!amount || isNaN(parseFloat(amount)) || parseFloat(amount) <= 0) {
      return res.status(400).json({
        error: "Invalid amount. Amount must be a positive number."
      });
    }
    if (!currency) {
      return res.status(400).json({ error: "Invalid currency. Currency is required." });
    }
    if (!intent) {
      return res.status(400).json({ error: "Invalid intent. Intent is required." });
    }
    const collect = {
      body: {
        intent: "CAPTURE",
        purchaseUnits: [
          {
            amount: {
              currencyCode: currency,
              value: amount
            }
          }
        ]
      },
      prefer: "return=minimal"
    };
    const { body, ...httpResponse } = await ordersController.createOrder(collect);
    const jsonResponse = JSON.parse(String(body));
    const httpStatusCode = httpResponse.statusCode;
    res.status(httpStatusCode).json(jsonResponse);
  } catch (error) {
    console.error("Failed to create order:", error);
    res.status(500).json({ error: "Failed to create order." });
  }
}
async function capturePaypalOrder(req, res) {
  try {
    const { orderID } = req.params;
    const collect = {
      id: orderID,
      prefer: "return=minimal"
    };
    const { body, ...httpResponse } = await ordersController.captureOrder(collect);
    const jsonResponse = JSON.parse(String(body));
    const httpStatusCode = httpResponse.statusCode;
    res.status(httpStatusCode).json(jsonResponse);
  } catch (error) {
    console.error("Failed to create order:", error);
    res.status(500).json({ error: "Failed to capture order." });
  }
}
async function loadPaypalDefault(req, res) {
  try {
    const clientToken = await getClientToken();
    res.json({
      clientToken
    });
  } catch (error) {
    console.error("PayPal authentication failed:", error);
    res.status(500).json({
      error: "PayPal configuration error",
      message: "Please check PayPal credentials"
    });
  }
}

// server/auth.ts
import bcrypt from "bcrypt";
async function hashPassword(password) {
  return await bcrypt.hash(password, 12);
}
async function verifyPassword(password, hashedPassword) {
  return await bcrypt.compare(password, hashedPassword);
}

// server/emailService.ts
import nodemailer from "nodemailer";
var transporter = nodemailer.createTransport({
  host: "mail.spacemail.com",
  port: 465,
  secure: true,
  // Use SSL for port 465
  tls: {
    rejectUnauthorized: false
  },
  auth: {
    user: "sales@futurewith.co",
    pass: "Mohammedamgfd24$$"
  }
});
function generateVerificationCode() {
  return Math.random().toString(36).substring(2, 8).toUpperCase();
}
function generateResetCode() {
  return Math.random().toString(36).substring(2, 8).toUpperCase();
}
async function sendVerificationEmail(email, code, username) {
  try {
    const mailOptions = {
      from: "sales@futurewith.co",
      to: email,
      subject: "Verify Your Baya3 Account",
      html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="text-align: center; margin-bottom: 30px;">
            <h1 style="color: #2C3E50; margin: 0;">Welcome to Baya3!</h1>
          </div>
          
          <div style="background-color: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
            <h2 style="color: #2C3E50; margin-top: 0;">Hi ${username},</h2>
            <p style="color: #555; line-height: 1.6;">
              Thank you for joining Baya3! To complete your registration, please use the verification code below:
            </p>
            
            <div style="text-align: center; margin: 30px 0;">
              <div style="background-color: #3498DB; color: white; font-size: 32px; font-weight: bold; padding: 15px 30px; border-radius: 8px; letter-spacing: 3px; display: inline-block;">
                ${code}
              </div>
            </div>
            
            <p style="color: #555; line-height: 1.6;">
              Enter this code on the verification page to activate your account. This code will expire in 24 hours.
            </p>
          </div>
          
          <div style="border-top: 1px solid #eee; padding-top: 20px; margin-top: 30px;">
            <p style="color: #999; font-size: 14px; margin: 0;">
              If you didn't create an account with Baya3, please ignore this email.
            </p>
            <p style="color: #999; font-size: 14px; margin: 10px 0 0 0;">
              Best regards,<br>
              The Baya3 Team
            </p>
          </div>
        </div>
      `,
      text: `Hi ${username},

Welcome to Baya3! Your verification code is: ${code}

Enter this code to verify your account. This code will expire in 24 hours.

If you didn't create an account with Baya3, please ignore this email.

Best regards,
The Baya3 Team`
    };
    await transporter.sendMail(mailOptions);
    console.log(`Verification email sent to ${email} with code: ${code}`);
    return true;
  } catch (error) {
    console.error("Error sending verification email:", error);
    return false;
  }
}
async function sendPasswordResetEmail(email, token, resetLink) {
  try {
    const resetUrl = resetLink || `https://your-domain.com/reset-password?token=${token}`;
    const mailOptions = {
      from: "sales@futurewith.co",
      to: email,
      subject: "Reset Your Baya3 Password",
      html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="text-align: center; margin-bottom: 30px;">
            <h1 style="color: #2C3E50; margin: 0;">Password Reset</h1>
          </div>
          
          <div style="background-color: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
            <p style="color: #555; line-height: 1.6;">
              You requested to reset your password for your Baya3 account. Click the button below to reset your password:
            </p>
            
            <div style="text-align: center; margin: 30px 0;">
              <a href="${resetUrl}" style="background-color: #E74C3C; color: white; text-decoration: none; font-size: 18px; font-weight: bold; padding: 15px 30px; border-radius: 8px; display: inline-block;">
                Reset Password
              </a>
            </div>
            
            <p style="color: #555; line-height: 1.6;">
              If the button doesn't work, copy and paste this link into your browser:
            </p>
            <p style="color: #3498DB; word-break: break-all; background-color: #f0f0f0; padding: 10px; border-radius: 4px; font-family: monospace;">
              ${resetUrl}
            </p>
            
            <p style="color: #555; line-height: 1.6;">
              This reset link will expire in 1 hour for security reasons.
            </p>
          </div>
          
          <div style="border-top: 1px solid #eee; padding-top: 20px; margin-top: 30px;">
            <p style="color: #999; font-size: 14px; margin: 0;">
              If you didn't request a password reset, please ignore this email.
            </p>
            <p style="color: #999; font-size: 14px; margin: 10px 0 0 0;">
              Best regards,<br>
              The Baya3 Team
            </p>
          </div>
        </div>
      `,
      text: `You requested to reset your password for your Baya3 account.

Click this link to reset your password: ${resetUrl}

This link will expire in 1 hour.

If you didn't request this reset, please ignore this email.

Best regards,
The Baya3 Team`
    };
    await transporter.sendMail(mailOptions);
    console.log(`Password reset email sent to ${email} with reset link`);
    return true;
  } catch (error) {
    console.error("Error sending password reset email:", error);
    return false;
  }
}

// server/routes.ts
async function registerRoutes(app2) {
  app2.use(session({
    secret: "baya3-secret-key-change-in-production",
    resave: false,
    saveUninitialized: false,
    cookie: {
      secure: false,
      // Set to true in production with HTTPS
      httpOnly: true,
      maxAge: 24 * 60 * 60 * 1e3
      // 24 hours
    }
  }));
  app2.get("/api/paypal/setup", async (req, res) => {
    await loadPaypalDefault(req, res);
  });
  app2.post("/api/paypal/order", async (req, res) => {
    await createPaypalOrder(req, res);
  });
  app2.post("/api/paypal/order/:orderID/capture", async (req, res) => {
    await capturePaypalOrder(req, res);
  });
  app2.post("/api/auth/register", async (req, res) => {
    try {
      const userData = registerSchema.parse(req.body);
      const existingUser = await storage.getUserByEmail(userData.email);
      if (existingUser) {
        if (existingUser.isVerified) {
          return res.status(400).json({ message: "Email already registered" });
        } else {
          const verificationCode2 = generateVerificationCode();
          const verificationCodeExpiry = new Date(Date.now() + 24 * 60 * 60 * 1e3);
          await storage.updateUser(existingUser.id, {
            verificationCode: verificationCode2,
            verificationCodeExpiry
          });
          const emailSent2 = await sendVerificationEmail(userData.email, verificationCode2, existingUser.username);
          if (!emailSent2) {
            return res.status(500).json({ message: "Failed to send verification email" });
          }
          console.log(`Verification email sent to ${userData.email} with code: ${verificationCode2}`);
          return res.status(200).json({
            message: "Registration successful. Please check your email for verification code.",
            requiresVerification: true
          });
        }
      }
      const existingUsername = await storage.getUserByUsername(userData.username);
      if (existingUsername) {
        return res.status(400).json({ message: "Username already taken" });
      }
      const hashedPassword = await hashPassword(userData.password);
      const verificationCode = generateVerificationCode();
      const verificationExpiry = new Date(Date.now() + 24 * 60 * 60 * 1e3);
      const user = await storage.createUser({
        username: userData.username,
        email: userData.email,
        password: hashedPassword,
        isVerified: false,
        verificationCode,
        verificationCodeExpiry: verificationExpiry
        // removed createdAt, let DB default handle it
      });
      const emailSent = await sendVerificationEmail(userData.email, verificationCode, userData.username);
      if (!emailSent) {
        console.error("Failed to send verification email");
      }
      res.status(201).json({
        message: "Registration successful. Please check your email for verification code.",
        userId: user.id
      });
    } catch (error) {
      console.error("Registration error:", error);
      res.status(400).json({ message: "Registration failed" });
    }
  });
  app2.post("/api/auth/verify", async (req, res) => {
    try {
      const { email, code } = req.body;
      if (!email || !code) {
        return res.status(400).json({ message: "Email and verification code are required" });
      }
      const user = await storage.getUserByEmail(email);
      if (!user || user.verificationCode !== code) {
        return res.status(400).json({ message: "Invalid verification code" });
      }
      if (user.verificationCodeExpiry && /* @__PURE__ */ new Date() > user.verificationCodeExpiry) {
        return res.status(400).json({ message: "Verification code has expired" });
      }
      await storage.updateUser(user.id, {
        isVerified: true,
        verificationCode: null,
        verificationCodeExpiry: null
      });
      res.json({ message: "Email verified successfully" });
    } catch (error) {
      console.error("Verification error:", error);
      res.status(400).json({ message: "Verification failed" });
    }
  });
  app2.post("/api/auth/resend-verification", async (req, res) => {
    try {
      const { email } = req.body;
      const user = await storage.getUserByEmail(email);
      if (!user) {
        return res.status(400).json({ message: "User not found" });
      }
      if (user.isVerified) {
        return res.status(400).json({ message: "Email already verified" });
      }
      const verificationCode = generateVerificationCode();
      const verificationCodeExpiry = new Date(Date.now() + 24 * 60 * 60 * 1e3);
      await storage.updateUser(user.id, {
        verificationCode,
        verificationCodeExpiry
      });
      const emailSent = await sendVerificationEmail(email, verificationCode, user.username);
      if (!emailSent) {
        return res.status(500).json({ message: "Failed to send verification email" });
      }
      console.log(`Verification email sent to ${email} with code: ${verificationCode}`);
      res.json({ message: "Verification code sent successfully" });
    } catch (error) {
      console.error("Resend verification error:", error);
      res.status(500).json({ message: "Server error" });
    }
  });
  app2.post("/api/auth/login", async (req, res) => {
    try {
      const loginData = loginSchema.parse(req.body);
      const user = await storage.getUserByEmail(loginData.email);
      if (!user) {
        return res.status(401).json({ message: "Invalid credentials" });
      }
      if (!user.isVerified) {
        return res.status(401).json({ message: "Please verify your email before logging in" });
      }
      const isValidPassword = await verifyPassword(loginData.password, user.password);
      if (!isValidPassword) {
        return res.status(401).json({ message: "Invalid credentials" });
      }
      req.session.userId = user.id;
      res.json({
        message: "Login successful",
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
          isAdmin: user.isAdmin
        }
      });
    } catch (error) {
      console.error("Login error:", error);
      res.status(400).json({ message: "Login failed" });
    }
  });
  app2.get("/api/auth/user", async (req, res) => {
    try {
      const userId = req.session?.userId;
      if (!userId) {
        return res.status(401).json({ message: "Not authenticated" });
      }
      const user = await storage.getUser(userId);
      if (!user) {
        return res.status(401).json({ message: "User not found" });
      }
      res.json({
        id: user.id,
        username: user.username,
        email: user.email,
        isAdmin: user.isAdmin,
        createdAt: user.createdAt
      });
    } catch (error) {
      console.error("Auth check error:", error);
      res.status(500).json({ message: "Authentication check failed" });
    }
  });
  app2.post("/api/auth/admin-login", async (req, res) => {
    try {
      const { email, password } = req.body;
      if (!email || !password) {
        return res.status(400).json({ message: "Email and password are required" });
      }
      const user = await storage.getUserByEmail(email);
      if (!user || !user.isAdmin) {
        return res.status(401).json({ message: "Invalid admin credentials" });
      }
      const isValidPassword = await verifyPassword(password, user.password);
      if (!isValidPassword) {
        return res.status(401).json({ message: "Invalid admin credentials" });
      }
      req.session.userId = user.id;
      res.json({
        message: "Admin login successful",
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
          isAdmin: user.isAdmin
        }
      });
    } catch (error) {
      console.error("Admin login error:", error);
      res.status(400).json({ message: "Admin login failed" });
    }
  });
  app2.post("/api/auth/logout", (req, res) => {
    req.session.destroy((err) => {
      if (err) {
        return res.status(500).json({ message: "Logout failed" });
      }
      res.clearCookie("connect.sid");
      res.json({ message: "Logged out successfully" });
    });
  });
  app2.post("/api/auth/forgot-password", async (req, res) => {
    try {
      const { email } = req.body;
      if (!email) {
        return res.status(400).json({ error: "Email is required" });
      }
      const user = await storage.getUserByEmail(email);
      if (!user) {
        return res.json({ message: "If an account with that email exists, a password reset link has been sent." });
      }
      const resetToken = generateResetCode();
      await storage.updateUser(user.id, {
        resetCode: resetToken
      });
      const resetLink = `${req.protocol}://${req.get("host")}/reset-password?token=${resetToken}`;
      const emailSent = await sendPasswordResetEmail(user.email, resetToken, resetLink);
      if (!emailSent) {
        return res.status(500).json({ error: "Failed to send password reset email" });
      }
      res.json({ message: "If an account with that email exists, a password reset link has been sent." });
    } catch (error) {
      console.error("Error in forgot password:", error);
      res.status(500).json({ error: "Failed to process password reset request" });
    }
  });
  app2.post("/api/auth/request-password-reset", async (req, res) => {
    const userId = req.session?.userId;
    if (!userId) {
      return res.status(401).json({ error: "Not authenticated" });
    }
    try {
      const { currentPassword } = req.body;
      if (!currentPassword) {
        return res.status(400).json({ error: "Current password is required" });
      }
      const user = await storage.getUser(userId);
      if (!user) {
        return res.status(404).json({ error: "User not found" });
      }
      const isCurrentPasswordValid = await verifyPassword(currentPassword, user.password);
      if (!isCurrentPasswordValid) {
        return res.status(400).json({ error: "Current password is incorrect" });
      }
      const resetToken = generateResetCode();
      await storage.updateUser(userId, {
        resetCode: resetToken
      });
      const resetLink = `${req.protocol}://${req.get("host")}/reset-password?token=${resetToken}`;
      const emailSent = await sendPasswordResetEmail(user.email, resetToken, resetLink);
      if (!emailSent) {
        return res.status(500).json({ error: "Failed to send password reset email" });
      }
      res.json({ message: "Password reset link sent to your email. Please check your inbox." });
    } catch (error) {
      console.error("Error requesting password reset:", error);
      res.status(500).json({ error: "Failed to request password reset" });
    }
  });
  app2.post("/api/auth/reset-password", async (req, res) => {
    try {
      const { token, newPassword } = req.body;
      if (!token || !newPassword) {
        return res.status(400).json({ error: "Reset token and new password are required" });
      }
      if (newPassword.length < 6) {
        return res.status(400).json({ error: "New password must be at least 6 characters" });
      }
      const user = await storage.getUserByResetCode(token);
      if (!user) {
        return res.status(400).json({ error: "Invalid or expired reset token" });
      }
      const hashedPassword = await hashPassword(newPassword);
      await storage.updateUser(user.id, {
        password: hashedPassword,
        resetCode: null
        // Clear the reset token
      });
      res.json({ message: "Password updated successfully. You can now login with your new password." });
    } catch (error) {
      console.error("Error resetting password:", error);
      res.status(500).json({ error: "Failed to reset password" });
    }
  });
  app2.get("/api/orders/my", async (req, res) => {
    const userId = req.session?.userId;
    if (!userId) {
      return res.status(401).json({ error: "Not authenticated" });
    }
    try {
      const userId2 = req.session.userId;
      const orders2 = await storage.getUserOrders(userId2);
      res.json(orders2);
    } catch (error) {
      console.error("Error fetching user orders:", error);
      res.status(500).json({ error: "Failed to fetch orders" });
    }
  });
  app2.get("/api/categories", async (req, res) => {
    try {
      const categories2 = await storage.getCategories();
      res.json(categories2);
    } catch (error) {
      res.status(500).json({ error: "Failed to fetch categories" });
    }
  });
  app2.get("/api/products", async (req, res) => {
    try {
      const { category } = req.query;
      let products2;
      if (category) {
        const categoryRecord = await storage.getCategoryBySlug(category);
        if (!categoryRecord) {
          return res.status(404).json({ error: "Category not found" });
        }
        products2 = await storage.getProductsByCategory(categoryRecord.id);
      } else {
        products2 = await storage.getProducts();
      }
      res.json(products2);
    } catch (error) {
      res.status(500).json({ error: "Failed to fetch products" });
    }
  });
  app2.get("/api/products/featured", async (req, res) => {
    try {
      const products2 = await storage.getFeaturedProducts();
      res.json(products2);
    } catch (error) {
      res.status(500).json({ error: "Failed to fetch featured products" });
    }
  });
  app2.get("/api/products/search", async (req, res) => {
    try {
      const { q } = req.query;
      if (!q || typeof q !== "string" || q.length < 2) {
        return res.json([]);
      }
      const products2 = await storage.getProducts();
      const searchResults = products2.filter(
        (product) => product.name.toLowerCase().includes(q.toLowerCase()) || product.description.toLowerCase().includes(q.toLowerCase())
      ).slice(0, 5);
      res.json(searchResults);
    } catch (error) {
      res.status(500).json({ error: "Failed to search products" });
    }
  });
  app2.get("/api/products/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const product = await storage.getProduct(id);
      if (!product) {
        return res.status(404).json({ error: "Product not found" });
      }
      res.json(product);
    } catch (error) {
      res.status(500).json({ error: "Failed to fetch product" });
    }
  });
  app2.post("/api/admin/products", async (req, res) => {
    try {
      const productData = insertProductSchema.parse(req.body);
      const product = await storage.createProduct(productData);
      res.status(201).json(product);
    } catch (error) {
      res.status(400).json({ error: "Invalid product data" });
    }
  });
  app2.put("/api/admin/products/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const updates = insertProductSchema.partial().parse(req.body);
      const product = await storage.updateProduct(id, updates);
      if (!product) {
        return res.status(404).json({ error: "Product not found" });
      }
      res.json(product);
    } catch (error) {
      res.status(400).json({ error: "Invalid product data" });
    }
  });
  app2.delete("/api/admin/products/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const deleted = await storage.deleteProduct(id);
      if (!deleted) {
        return res.status(404).json({ error: "Product not found" });
      }
      res.status(204).send();
    } catch (error) {
      res.status(500).json({ error: "Failed to delete product" });
    }
  });
  app2.get("/api/cart/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const cartItems2 = await storage.getCartItems(sessionId);
      res.json(cartItems2);
    } catch (error) {
      res.status(500).json({ error: "Failed to fetch cart items" });
    }
  });
  app2.post("/api/cart", async (req, res) => {
    try {
      const cartItemData = insertCartItemSchema.parse(req.body);
      const cartItem = await storage.addToCart(cartItemData);
      res.status(201).json(cartItem);
    } catch (error) {
      res.status(400).json({ error: "Invalid cart item data" });
    }
  });
  app2.put("/api/cart/:sessionId/:productId", async (req, res) => {
    try {
      const { sessionId, productId } = req.params;
      const { quantity } = req.body;
      if (typeof quantity !== "number" || quantity < 0) {
        return res.status(400).json({ error: "Invalid quantity" });
      }
      const cartItem = await storage.updateCartItemQuantity(sessionId, parseInt(productId), quantity);
      res.json(cartItem);
    } catch (error) {
      res.status(500).json({ error: "Failed to update cart item" });
    }
  });
  app2.delete("/api/cart/:sessionId/:productId", async (req, res) => {
    try {
      const { sessionId, productId } = req.params;
      const removed = await storage.removeFromCart(sessionId, parseInt(productId));
      if (!removed) {
        return res.status(404).json({ error: "Cart item not found" });
      }
      res.status(204).send();
    } catch (error) {
      res.status(500).json({ error: "Failed to remove cart item" });
    }
  });
  app2.delete("/api/cart/:sessionId", async (req, res) => {
    try {
      const { sessionId } = req.params;
      await storage.clearCart(sessionId);
      res.status(204).send();
    } catch (error) {
      res.status(500).json({ error: "Failed to clear cart" });
    }
  });
  app2.post("/api/cart/save-to-user", async (req, res) => {
    try {
      const { userId } = req.body;
      if (!userId) {
        return res.status(400).json({ error: "User ID is required" });
      }
      const guestItems = await storage.getCartItems("guest");
      for (const item of guestItems) {
        await storage.addToCart({
          sessionId: `user_${userId}`,
          productId: item.productId,
          quantity: item.quantity
        });
      }
      await storage.clearCart("guest");
      res.json({ message: "Cart saved to user successfully" });
    } catch (error) {
      console.error("Error saving cart to user:", error);
      res.status(500).json({ error: "Failed to save cart to user" });
    }
  });
  app2.post("/api/orders", async (req, res) => {
    try {
      const { customerEmail, customerPhone, totalUSD, currency, items } = req.body;
      if (!customerEmail || !totalUSD || !currency || !items) {
        return res.status(400).json({ error: "Missing required order data" });
      }
      const order = await storage.createOrder({
        customerEmail,
        customerPhone,
        totalUSD,
        currency,
        status: "pending"
      });
      for (const item of items) {
        await storage.createOrderItem({
          orderId: order.id,
          productId: item.product.id,
          quantity: item.quantity,
          priceUSD: item.product.priceUSD,
          pubgPlayerId: item.pubgPlayerId
        });
      }
      const hasPubgProducts = items.some(
        (item) => item.product.name.toLowerCase().includes("pubg") && item.pubgPlayerId
      );
      if (hasPubgProducts) {
        try {
          const nodemailer2 = __require("nodemailer");
          const pubgItems = items.filter(
            (item) => item.product.name.toLowerCase().includes("pubg") && item.pubgPlayerId
          );
          const emailContent = `
New PUBG UC Order - Order #${order.id}

Customer Details:
- Email: ${customerEmail}
- Phone: ${customerPhone || "Not provided"}

PUBG Products:
${pubgItems.map((item) => `
- Product: ${item.product.name}
- Quantity: ${item.quantity}
- PUBG Player ID: ${item.pubgPlayerId}
- Price: $${parseFloat(item.product.priceUSD).toFixed(2)} USD
`).join("")}

Total: $${totalUSD} USD (${currency})

Please process this PUBG UC delivery manually.
          `;
          const transporter2 = nodemailer2.createTransporter({
            service: "gmail",
            auth: {
              user: "notification@baya3.com",
              pass: "your-email-password"
              // This would be configured properly
            }
          });
          await transporter2.sendMail({
            from: "notification@baya3.com",
            to: "moe.eryani2424@gmail.com",
            subject: `New PUBG UC Order #${order.id}`,
            text: emailContent
          });
        } catch (emailError) {
          console.error("Failed to send PUBG order notification:", emailError);
        }
      }
      res.status(201).json(order);
    } catch (error) {
      console.error("Order creation error:", error);
      res.status(400).json({ error: "Failed to create order" });
    }
  });
  app2.get("/api/orders/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const order = await storage.getOrder(id);
      if (!order) {
        return res.status(404).json({ error: "Order not found" });
      }
      res.json(order);
    } catch (error) {
      res.status(500).json({ error: "Failed to fetch order" });
    }
  });
  app2.get("/api/admin/orders", async (req, res) => {
    try {
      const orders2 = await storage.getOrders();
      res.json(orders2);
    } catch (error) {
      res.status(500).json({ error: "Failed to fetch orders" });
    }
  });
  app2.put("/api/admin/orders/:id/status", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const { status, paypalOrderId } = req.body;
      const order = await storage.updateOrderStatus(id, status, paypalOrderId);
      if (!order) {
        return res.status(404).json({ error: "Order not found" });
      }
      res.json(order);
    } catch (error) {
      res.status(500).json({ error: "Failed to update order status" });
    }
  });
  const httpServer = createServer(app2);
  return httpServer;
}

// server/index.ts
var app = express2();
app.use(express2.json());
app.use(express2.urlencoded({ extended: false }));
var log2 = () => {
};
app.use((req, res, next) => {
  const start = Date.now();
  const path3 = req.path;
  let capturedJsonResponse = void 0;
  const originalResJson = res.json;
  res.json = function(bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };
  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path3.startsWith("/api")) {
      let logLine = `${req.method} ${path3} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }
      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "\u2026";
      }
      log2(logLine);
    }
  });
  next();
});
(async () => {
  const server = await registerRoutes(app);
  app.use((err, _req, res, _next) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";
    res.status(status).json({ message });
    throw err;
  });
  if (app.get("env") === "development") {
    const { setupVite: setupVite2, log: viteLog } = await Promise.resolve().then(() => (init_vite(), vite_exports));
    log2 = viteLog;
    await setupVite2(app, server);
  } else {
    const { serveStatic: serveStatic2, log: viteLog } = await Promise.resolve().then(() => (init_vite(), vite_exports));
    log2 = viteLog;
    serveStatic2(app);
  }
  const port = Number(process.env.PORT) || 3e3;
  const host = "0.0.0.0";
  server.listen(port, host, () => {
    log2(`serving on http://${host}:${port}`);
  });
})();
